﻿
#include "IOT_Gyro.h"
#include "Arduino.h"

//------------------------------------------------------------------------------------//
#if  defined(GYRO_USE)

// I2Cdev and MPU6050 must be installed as libraries, or else the .cpp/.h files
// for both classes must be in the include path of your project
#include "I2Cdev.h"
#include "MPU6050_6Axis_MotionApps20.h"
#if I2CDEV_IMPLEMENTATION == I2CDEV_ARDUINO_WIRE
#include "Wire.h"
#endif

MPU6050 mpu;

#endif

//----------------------------------- Gyro --------------------------------------------//
#if  defined(GYRO_USE)

// MPU control/status vars
uint8_t mpuIntStatus;     // holds actual interrupt status byte from MPU
uint8_t devStatus;        // return status after each device operation (0 = success, !0 = error)
uint16_t packetSize;      // expected DMP packet size (default is 42 bytes)
uint16_t fifoCount;       // count of all bytes currently in FIFO
uint8_t fifoBuffer[64];   // FIFO storage buffer
Quaternion q;             // [w, x, y, z]         quaternion container
VectorFloat gravity;      // [x, y, z]            gravity vector

VectorInt16 aa;         // [x, y, z]            accel sensor measurements
VectorInt16 aaReal;     // [x, y, z]            gravity-free accel sensor measurements
VectorInt16 gg;         // [x, y, z]            accel sensor measurements

VectorInt16 aaWorld;    // [x, y, z]            world-frame accel sensor measurements
//float euler[3];         // [psi, theta, phi]    Euler angle container
float ypr[3];           // [yaw, pitch, roll]   yaw/pitch/roll container and gravity vector



int16_t setXAccelOffset;
int16_t setYAccelOffset;
int16_t setZAccelOffset;

int16_t setXGyroOffset;
int16_t setYGyroOffset;
int16_t setZGyroOffset;


#endif


//------------------------------------------------------------------------------------//
void GyroInit()
{

  // join I2C bus (I2Cdev library doesn't do this automatically)
#if I2CDEV_IMPLEMENTATION == I2CDEV_ARDUINO_WIRE
  Wire.begin();
  Wire.setClock(400000); // 400kHz I2C clock. Comment this line if having compilation difficulties
#elif I2CDEV_IMPLEMENTATION == I2CDEV_BUILTIN_FASTWIRE
  Fastwire::setup(400, true);
#endif

  mpu.initialize();
  //Serial.println(mpu.testConnection() ? F("MPU6050 connection successful") : F("MPU6050 connection failed"));
  devStatus = mpu.dmpInitialize();

  // supply your own gyro offsets here, scaled for min sensitivity
  // Apply calibration value
  mpu.setXAccelOffset(setXAccelOffset);
  mpu.setYAccelOffset(setYAccelOffset);
  mpu.setZAccelOffset(setZAccelOffset);

  mpu.setXGyroOffset(setXGyroOffset);
  mpu.setYGyroOffset(setYGyroOffset);
  mpu.setZGyroOffset(setZGyroOffset);

  if (devStatus == 0) // make sure it worked (returns 0 if so)
  {
    //   Serial.println(F("Enabling DMP...")); // turn on the DMP, now that it's ready
    mpu.setDMPEnabled(true);
    mpuIntStatus = mpu.getIntStatus();
    // set our DMP Ready flag so the main loop() function knows it's okay to use it
    // Serial.println(F("DMP ready! Waiting for first interrupt..."));
    packetSize = mpu.dmpGetFIFOPacketSize();  // get expected DMP packet size for later comparison
  }
  else
  {
    // ERROR!
    // 1 = initial memory load failed
    // 2 = DMP configuration updates failed
    // (if it's going to break, usually the code will be 1)
    Serial.print(F("DMP Initialization failed (code "));
    Serial.print(devStatus);
    Serial.println(F(")"));
  }
}


int16_t getGyro(int gyro)
{
#if  defined(GYRO_USE)
  if (gyro == X) return gg.x ;
  else if (gyro == Y) return gg.y;
  else if (gyro == Z) return gg.z;
  else return 0;
#endif
}

int16_t getACC(int acc)
{
#if  defined(GYRO_USE)
  if (acc == X) return aa.x ;
  else if (acc == Y) return aa.y;
  else if (acc == Z) return aa.z;
  else return 0;
#endif
}

int16_t eulerAngle(int euler)
{
	
	/*
	if (euler == YAW) Serial.print(ypr[0]);  
	else if (euler == PITCH)  Serial.print(ypr[1]); 
	else if (euler == ROLL) Serial.print(ypr[2]);
	
	Serial.print("\t");
	*/
	
  if (euler == YAW) return (ypr[0] * 180) / M_PI ;
  else if (euler == PITCH) return (ypr[1] * 180) / M_PI;
  else if (euler == ROLL) return (ypr[2] * 180) / M_PI;
  else return 0;
}

void GyroCalibration()
{
  GyroCalibrationProcess(1);
  GyroInit();
  delay(20);
}

void GyroCalibrationProcess(int tuneCount = 1)
{

#if I2CDEV_IMPLEMENTATION == I2CDEV_ARDUINO_WIRE
  Wire.begin();
#elif I2CDEV_IMPLEMENTATION == I2CDEV_BUILTIN_FASTWIRE
  Fastwire::setup(400, true);
#endif

  mpu.initialize();

  // Serial.println("PID tuning Each Dot = 100 readings");
  mpu.CalibrateAccel(tuneCount);
  mpu.CalibrateGyro(tuneCount);

  uint8_t devAddr = 0x68;
  uint8_t AOffsetRegister = (mpu.getDeviceID() < 0x38 ) ? MPU6050_RA_XA_OFFS_H : 0x77;
  int16_t Data[3];

  if (AOffsetRegister == 0x06) I2Cdev::readWords(devAddr, AOffsetRegister, 3, (uint16_t *)Data);
  else {
    I2Cdev::readWords(devAddr, AOffsetRegister, 1, (uint16_t *)Data);
    I2Cdev::readWords(devAddr, AOffsetRegister + 3, 1, (uint16_t *)Data + 1);
    I2Cdev::readWords(devAddr, AOffsetRegister + 6, 1, (uint16_t *)Data + 2);
  }

  setXAccelOffset = Data[0];
  setYAccelOffset = Data[1];
  setZAccelOffset = Data[2];

  I2Cdev::readWords(devAddr, 0x13, 3, (uint16_t *)Data);

  setXGyroOffset = Data[0];
  setYGyroOffset = Data[1];
  setZGyroOffset = Data[2];

}


boolean GyroRead()
{
  boolean success = false;

  // get current FIFO count
  fifoCount = mpu.getFIFOCount();

  // check for overflow (this should never happen unless our code is too inefficient)
  if (fifoCount == 1024)
  {
    // reset so we can continue cleanly
    mpu.resetFIFO();
    // Serial.println(F("FIFO overflow!"));
    // otherwise, check for DMP data ready interrupt (this should happen frequently)
  }
  else
  {
    while (fifoCount >= packetSize) { // Lets catch up to NOW, someone is using the dreaded delay()!
      mpu.getFIFOBytes(fifoBuffer, packetSize);
      // track FIFO count here in case there is > 1 packet available
      // (this lets us immediately read more without waiting for an interrupt)
      fifoCount -= packetSize;
    }

    // display quaternion values in easy matrix form: w x y z
    mpu.dmpGetQuaternion(&q, fifoBuffer);

    //angle
    // display Euler angles in degrees
    mpu.dmpGetQuaternion(&q, fifoBuffer);
    mpu.dmpGetGravity(&gravity, &q);
    mpu.dmpGetYawPitchRoll(ypr, &q, &gravity);

    //acc
    //  mpu.dmpGetQuaternion(&q, fifoBuffer);
    mpu.dmpGetAccel(&aa, fifoBuffer);
    mpu.dmpGetGravity(&gravity, &q);
    mpu.dmpGetLinearAccel(&aaReal, &aa, &gravity);

    //GYRO
    mpu.dmpGetGyro(&gg, fifoBuffer);


    success = true;
  }

  return success;
}


void delayForGyro(float time)
{
  unsigned long startTimeDrive = millis();
  while (millis() < startTimeDrive + (time * 1000))
  {
    while (!GyroRead());
  }
}